/*
 * Modified by Michael Otte, MIT, 2013
 *
 * USB retina driver
 * Martin Ebner, IGI / TU Graz (ebner at igi.tugraz.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 * This driver is based on the 2.6.3 version of drivers/usb/usb-skeleton.c
 * (Copyright (C) 2001-2004 Greg Kroah-Hartman (greg@kroah.com))
 *
 */

// Need to add the following line to /etc/udev/rules.d/62-fx2.rules
// KERNEL=="retina[0-9]", NAME="retina0", SYMLINK+="retina0", MODE="0666"
//
// Notes: java code used "byte" instead of "signed char"
// java "a >>> b" is logical right shift, in C this is posible as "(unsigned)a >> b"
// "a >> b" is arithmetic right shift and same in C and Java



#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>


#define READ_BUFFER_SIZE 10000

// this puts the most recent data into the data structure
void updateData(int file_desc, int** data)
{
  // constantes used to help extract and parse stream data
  static const short xmask = 0x00fe;
  static const signed char xshift = 1;
  static const short ymask = 0x7f00;
  static const signed char yshift = 8;
  static const short typemask = 1;
  static const int sizeX = 128;
  static const int sizeY = 128;
  int sxm = sizeX - 1;  

  static char buffer[READ_BUFFER_SIZE]; // the buffer used to read the stream

  // non constants
  static short TICK_US = 1;
  static int wrapAdd = 0;

  int bytesSent = read(file_desc,buffer, READ_BUFFER_SIZE); // read file

  if( bytesSent % 4 != 0 )
  {
    bytesSent = (bytesSent / 4) * 4;
  }

  // parse data
  int i;
  for(i = 0; i < bytesSent; i+=4)
  {
    if ((buffer[i + 3] & 0x80) == 0x80)
    {
      wrapAdd += 0x4000L;
    }
    else if ((buffer[i + 3] & 0x40) == 0x40) 
    {
      // this firmware version uses reset events to reset timestamps
    }
    else
    {
      int address = (int) ((buffer[i] & 0xFF) | ((buffer[i + 1] & 0xFF) << 8));
      int shortts = (buffer[i + 2] & 0xff | ((buffer[i + 3] & 0xff) << 8));
      int timestamp = (int) wrapAdd + (TICK_US * (shortts));
      signed char polarity = (1 - address & 1); // 0 = polarity on, 1 polarity off
      short x = (short) (sxm - ((short) ((unsigned)(address & xmask) >> xshift)));
      short y = (short) ((unsigned)(address & ymask) >> yshift);
      // printf("T:%d, p: %d, x:%d y:%d  \n", timestamp, (int)polarity, (int)x, (int)y); 
      data[y][x] = (int)polarity;
    }
  }
}

/*
// Example of how to use
int main(int argc, char *argv[])
{
  // open device file
  int file_desc;
  file_desc = open("/dev/retina0", O_RDWR);
  if (file_desc < 0) 
  {
    printf("Can't open device file");
    exit(-1);
  }

  // make structure to hold data
  int xPixels = 128;
  int yPixels = 128;
  int** data = (int**)calloc(yPixels, sizeof(int*));
  int i;
  for(i = 0; i < yPixels; i++)
  {
    data[i] = (int*)calloc(xPixels, sizeof(int));
  }

  // get data
  for(i = 0; i < 10000; i++)
  {
    updateData(file_desc, data);
  }

  // delete data
  for(i = 0; i < xPixels; i++)
  {
    free(data[i]);
  } 
  free(data);

  close(file_desc);
  return 0;
}*/

